// Slider functionality
let currentSlide = 0
const slides = document.querySelectorAll('.slide-stitch')
const dots = document.querySelectorAll('.dot-stitch')
const totalSlides = slides.length

function goToSlide(index) {
  if (slides.length === 0) return
  
  slides[currentSlide]?.classList.remove('active')
  dots[currentSlide]?.classList.remove('active')
  
  currentSlide = index
  if (currentSlide >= totalSlides) currentSlide = 0
  if (currentSlide < 0) currentSlide = totalSlides - 1
  
  slides[currentSlide]?.classList.add('active')
  dots[currentSlide]?.classList.add('active')
}

function goToNext() {
  goToSlide(currentSlide + 1)
}

function goToPrevious() {
  goToSlide(currentSlide - 1)
}

// Auto-slide
if (slides.length > 0) {
  setInterval(() => {
    goToNext()
  }, 3000)
  
  goToSlide(0)
}

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'index'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initIndexPage()
  }
}

// Index page functions
function initIndexPage() {
  loadCourses()
}

function loadCourses() {
  const saved = localStorage.getItem('sewingCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  } else {
    courses = [
      {
        id: 1,
        project: 'Платье в стиле ретро',
        teacher: 'Елена Волкова',
        completionDate: '2025-01-15',
        resultPhoto: 'images/result-1.jpg'
      },
      {
        id: 2,
        project: 'Жилет из джинсовой ткани',
        teacher: 'Мария Смирнова',
        completionDate: '2024-12-20',
        resultPhoto: 'images/result-2.jpg'
      },
      {
        id: 3,
        project: 'Сумка-тоут с вышивкой',
        teacher: 'Анна Петрова',
        completionDate: '2024-11-10',
        resultPhoto: 'images/result-3.jpg'
      }
    ]
    localStorage.setItem('sewingCourses', JSON.stringify(courses))
  }
  
  renderCourses(courses)
}

function renderCourses(courses) {
  const container = document.getElementById('projects-grid')
  if (!container) return
  
  if (courses.length === 0) {
    container.innerHTML = `
      <div class="empty-stitch" style="grid-column: 1 / -1;">
        <div class="empty-icon-stitch">
          <img src="icons/scissors.svg" alt="Empty" style="width: 64px; height: 64px;">
        </div>
        <h2 class="empty-title-stitch">Нет проектов</h2>
        <p class="empty-text-stitch">Начните свой творческий путь с выбора курса</p>
        <button class="empty-button-stitch" onclick="window.location.href='enroll.html'">
          Записаться
        </button>
      </div>
    `
    return
  }
  
  container.innerHTML = courses.map(course => {
    const date = course.completionDate 
      ? new Date(course.completionDate).toLocaleDateString('ru-RU', { 
          day: 'numeric', 
          month: 'long', 
          year: 'numeric' 
        })
      : 'В процессе'
    
    return `
      <div class="project-card-stitch">
        <div class="project-image-stitch">
          <img src="${course.resultPhoto}" alt="${course.project}">
          <div class="project-overlay-stitch">
            <div class="project-badge-stitch">
              <img src="icons/heart.svg" alt="Heart" style="width: 18px; height: 18px;">
              <span>Завершено</span>
            </div>
          </div>
        </div>
        <div class="project-content-stitch">
          <h3 class="project-title-stitch">${course.project}</h3>
          <div class="project-info-stitch">
            <div class="info-item-stitch">
              <img src="icons/user.svg" alt="User" style="width: 18px; height: 18px;">
              <span>${course.teacher}</span>
            </div>
            <div class="info-item-stitch">
              <img src="icons/calendar.svg" alt="Calendar" style="width: 18px; height: 18px;">
              <span>${date}</span>
            </div>
          </div>
        </div>
      </div>
    `
  }).join('')
}

// Enroll page functions
function initEnrollPage() {
  renderLevels()
  setupForm()
}

const levels = [
  { 
    id: 'basic', 
    name: 'Базовый', 
    icon: 'scissors',
    color: 'var(--terracotta)',
    description: 'Основы шитья для начинающих. Изучите базовые техники и создайте первые проекты.',
    features: ['Работа с тканью', 'Основные швы', 'Простая выкройка', 'Первый проект']
  },
  { 
    id: 'cutting', 
    name: 'Кройка', 
    icon: 'ruler',
    color: 'var(--dusty-rose)',
    description: 'Профессиональная работа с выкройками. Научитесь создавать и корректировать лекала.',
    features: ['Снятие мерок', 'Построение выкройки', 'Корректировка', 'Раскрой ткани']
  },
  { 
    id: 'modeling', 
    name: 'Моделирование', 
    icon: 'layers',
    color: 'var(--sage-green)',
    description: 'Создание сложных моделей одежды. Освойте техники моделирования и конструирования.',
    features: ['Моделирование', 'Конструирование', 'Сложные формы', 'Профессиональные техники']
  },
  { 
    id: 'author', 
    name: 'Авторская одежда', 
    icon: 'palette',
    color: 'var(--deep-burgundy)',
    description: 'Разработка уникальных дизайнов. Создавайте авторские коллекции и развивайте свой стиль.',
    features: ['Дизайн одежды', 'Авторские коллекции', 'Брендинг', 'Портфолио']
  }
]

function renderLevels() {
  const container = document.getElementById('levels-container')
  if (!container) return
  
  container.innerHTML = levels.map((level, index) => {
    return `
      <button 
        type="button" 
        class="level-card-stitch" 
        data-level="${level.id}"
        onclick="selectLevel('${level.id}')"
        style="border-color: var(--border-warm); background: var(--warm-cream);"
      >
        <div class="level-icon-stitch" style="background: var(--warm-beige); color: ${level.color};">
          <img src="icons/${level.icon}.svg" alt="${level.name}" style="width: 32px; height: 32px;">
        </div>
        <h3 class="level-name-stitch" style="color: ${level.color};">${level.name}</h3>
        <p class="level-desc-stitch">${level.description}</p>
        <div class="level-features-stitch">
          ${level.features.map(feature => `
            <span class="feature-tag-stitch" style="border-color: ${level.color};">
              ${feature}
            </span>
          `).join('')}
        </div>
      </button>
    `
  }).join('')
}

let selectedLevel = ''
let selectedDate = ''

function selectLevel(id) {
  selectedLevel = id
  
  // Update UI
  document.querySelectorAll('.level-card-stitch').forEach(card => {
    const levelId = card.getAttribute('data-level')
    const level = levels.find(l => l.id === levelId)
    
    if (levelId === id) {
      card.classList.add('selected')
      card.style.borderColor = level.color
      card.style.backgroundColor = `${level.color}15`
      
      // Add check indicator
      if (!card.querySelector('.check-mark-stitch')) {
        const check = document.createElement('div')
        check.className = 'check-mark-stitch'
        check.style.color = level.color
        check.innerHTML = `<img src="icons/check-circle.svg" alt="Check" style="width: 32px; height: 32px;">`
        card.appendChild(check)
      }
    } else {
      card.classList.remove('selected')
      card.style.borderColor = 'var(--border-warm)'
      card.style.backgroundColor = 'var(--warm-cream)'
      
      const check = card.querySelector('.check-mark-stitch')
      if (check) check.remove()
    }
  })
  
  // Show date section
  const dateSection = document.getElementById('date-section')
  if (dateSection) {
    dateSection.style.display = 'flex'
    dateSection.style.opacity = '1'
    dateSection.style.height = 'auto'
  }
  
  updateSubmitButton()
}

function setupForm() {
  const dateInput = document.getElementById('date-input')
  if (dateInput) {
    const today = new Date()
    const maxDate = new Date(today.getTime() + 180 * 24 * 60 * 60 * 1000)
    dateInput.min = today.toISOString().split('T')[0]
    dateInput.max = maxDate.toISOString().split('T')[0]
    
    dateInput.addEventListener('change', (e) => {
      selectedDate = e.target.value
      updateSubmitButton()
    })
  }
}

function updateSubmitButton() {
  const button = document.getElementById('submit-btn')
  if (button) {
    button.disabled = !selectedLevel || !selectedDate
  }
}

function handleSubmit(e) {
  if (e) e.preventDefault()
  
  if (!selectedLevel || !selectedDate) return
  
  const saved = localStorage.getItem('sewingCourses')
  const courses = saved ? JSON.parse(saved) : []
  
  const levelData = levels.find(l => l.id === selectedLevel)
  
  const newCourse = {
    id: Date.now(),
    project: `Проект: ${levelData.name}`,
    teacher: 'Преподаватель будет назначен',
    completionDate: null,
    resultPhoto: 'images/result-placeholder.jpg'
  }
  
  courses.push(newCourse)
  localStorage.setItem('sewingCourses', JSON.stringify(courses))
  
  // Show success
  const form = document.getElementById('enroll-form')
  if (form) {
    form.innerHTML = `
      <div class="success-stitch">
        <img src="icons/check-circle.svg" alt="Success" style="width: 40px; height: 40px;">
        <span>Запись оформлена! Переходим к вашим курсам...</span>
      </div>
    `
    
    setTimeout(() => {
      window.location.href = 'index.html'
    }, 2000)
  }
}

// Initialize on load
document.addEventListener('DOMContentLoaded', initPage)

